#ifndef _SSH_SSH_h
#define _SSH_SSH_h

#include <Core/Core.h>
#include <JobQueue/JobQueue.h>

#include <libssh2.h>
#include <libssh2_sftp.h>

namespace Upp {

INITIALIZE(SSH)

typedef LIBSSH2_SESSION SshSession;
typedef LIBSSH2_CHANNEL SshChannel;
typedef LIBSSH2_SFTP    SFtpSession;
typedef LIBSSH2_SFTP_HANDLE SFtpHandle;
typedef LIBSSH2_SFTP_ATTRIBUTES SFtpAttrs;

class SshSubsystem;
class SFtp;

class Ssh : private NoCopy, public JobQueue {
public:
    Ssh& Auth(const String& user, const String& pass)       { username = user; password = pass; return *this; }
    Ssh& Keys(const String& prikey, const String& pubkey)   { private_key = prikey; public_key = pubkey; return *this; }

    String               GetBanner() const                  { return session ? pick(String(libssh2_session_banner_get(session))) : Null; }
    const String&        GetFingerprint() const             { return fingerprint; }
    const Vector<String> GetAuthMethods() const             { return pick(Split(auth_methods, ',')); }
    SshSession*          GetSession() const                 { return session; }
    TcpSocket&           GetSocket()                        { return socket; }

    Ssh& StartConnect(const String& host, int port);
    Ssh& StartDisconnect();
    bool Connect(const String& host, int port, const String& user, const String& pass)  { Auth(user, pass).StartConnect(host, port); return Execute(); }
    bool Disconnect()                                                                   { StartDisconnect(); return Execute(); }

    Event<> WhenHash;
    Event<> WhenAuth;

private:
    TcpSocket   socket;
    IpAddrInfo  addrinfo;
    SshSession* session;
    String      fingerprint;
    String      auth_methods;
    String      username;
    String      password;
    String      private_key;
    String      public_key;
    friend class SshSubsystem;

    void CleanUp();
    void Error(int code = 0, const char* msg = NULL);
    bool WouldBlock()   { return session && libssh2_session_last_errno(session) == LIBSSH2_ERROR_EAGAIN; }

public:
    Ssh();
    virtual ~Ssh() { CleanUp(); }
};

class SshSubsystem : NoCopy, public JobQueue {
public:
    enum class Type                     { UNDEFINED, CHANNEL, SFTP, SCP, EXEC, SHELL };
    void Session(Ssh& session);
    void ChunkSize(int sz)              { chunk_size = max(sz, 4096); }
    Type GetType() const                { return type; }
    template <class T> T&   To()        { return *reinterpret_cast<T *>(this); }
    template <class T> bool Is() const  { return dynamic_cast<const T *>(this); }

    SshSubsystem();
    virtual ~SshSubsystem()             { ssh = NULL; }
  
protected:
    virtual void StartInit() = 0;
    virtual void StartStop() = 0;
    virtual void CleanUp()   = 0;

    void    Error(int code = 0, const char* msg = NULL);
    bool    WouldBlock() const          { ASSERT(ssh); return ssh->WouldBlock(); }

    Ssh*    ssh;
    Type    type;
    int     chunk_size;
    String  packet;
    int64   packet_length;
};

class SFtp : public SshSubsystem {
public:
    enum Flags {
        READ     = LIBSSH2_FXF_READ,
        WRITE    = LIBSSH2_FXF_WRITE,
        APPEND   = LIBSSH2_FXF_APPEND,
        CREATE   = LIBSSH2_FXF_CREAT,
        TRUNCATE = LIBSSH2_FXF_TRUNC,
        EXCLUDE  = LIBSSH2_FXF_EXCL
    };

    enum Permissions {
        IRUSR = LIBSSH2_SFTP_S_IRUSR,
        IWUSR = LIBSSH2_SFTP_S_IWUSR,
        IXUSR = LIBSSH2_SFTP_S_IXUSR,
        IRWXU = IRUSR | IWUSR | IXUSR,
        IRGRP = LIBSSH2_SFTP_S_IRGRP,
        IWGRP = LIBSSH2_SFTP_S_IWGRP,
        IXGRP = LIBSSH2_SFTP_S_IXGRP,
        IRWXG = IRGRP | IWGRP | IXGRP,
        IROTH = LIBSSH2_SFTP_S_IROTH,
        IWOTH = LIBSSH2_SFTP_S_IWOTH,
        IXOTH = LIBSSH2_SFTP_S_IXOTH,
        IRWXO = IROTH | IWOTH | IXOTH,
        IRWXA = IRWXU | IRWXG | IRWXO
    };

    class DirEntry : Moveable<DirEntry> {
        public:
            String GetName() const          { return file; }
            String GetEntry() const         { return entry; }
            int64  GetUid() const           { return a.flags & LIBSSH2_SFTP_ATTR_UIDGID ? a.uid : -1; }
            int64  GetGid() const           { return a.flags & LIBSSH2_SFTP_ATTR_UIDGID ? a.gid : -1; }
            int64  GetSize() const          { return a.flags & LIBSSH2_SFTP_ATTR_SIZE ? a.filesize : -1; }
            Time   GetLastModified() const  { return a.flags & LIBSSH2_SFTP_ATTR_ACMODTIME ? TimeFromUTC(a.mtime) : Null; }
            Time   GetLastAccessed() const  { return a.flags & LIBSSH2_SFTP_ATTR_ACMODTIME ? TimeFromUTC(a.atime) : Null; }
            SFtpAttrs& GetAttrs()           { return a; }

            bool IsFile() const             { return LIBSSH2_SFTP_S_ISREG(a.permissions); }
            bool IsDirectory() const        { return LIBSSH2_SFTP_S_ISDIR(a.permissions); }
            bool IsSymLink() const          { return LIBSSH2_SFTP_S_ISLNK(a.permissions); }
            bool IsSpecial() const          { return LIBSSH2_SFTP_S_ISCHR(a.permissions); }
            bool IsBlock() const            { return LIBSSH2_SFTP_S_ISBLK(a.permissions); }
            bool IsFifo() const             { return LIBSSH2_SFTP_S_ISFIFO(a.permissions); }
            bool IsSocket() const           { return LIBSSH2_SFTP_S_ISSOCK(a.permissions); }
            bool IsReadable() const         { return CanMode(IRUSR, IRGRP, IROTH); }
            bool IsWriteable() const        { return CanMode(IWUSR, IWGRP, IWOTH); }
            bool IsReadOnly() const         { return IsReadable() && !IsWriteable(); }
            bool IsExecutable() const       { return !IsDirectory() && CanMode(IXUSR, IXGRP, IXOTH); }

            DirEntry()                      { Clear(); }
            DirEntry(const Nuller&)         { Clear(); }

        private:
            bool CanMode(dword u, dword g, dword o) const;
            void Clear()                    { Zero(a); }
            String file;
            String entry;
            SFtpAttrs a;
            friend class SFtp;
    };
    typedef Vector<DirEntry> DirList;

public:
    SFtp& Handle(SFtpHandle* hndl)  { if(hndl) handle = hndl; return *this; }
    SFtpSession* GetSession() const { return session; }
    SFtpHandle*  GetHandle() const  { return handle; }

    SFtp& StartOpen(const String& path, unsigned long flags, long mode);
    SFtp& StartClose();
    SFtp& StartRemove(const String& path);
    SFtp& StartRename(const String& oldpath, const String& newpath);
    SFtp& StartSetStat(const SFtpAttrs& attrs)                                       { StartFStat(const_cast<SFtpAttrs&>(attrs), true); }
    SFtp& StartGetStat(SFtpAttrs& attrs)                                             { Zero(attrs); StartFStat(attrs, false); }
    SFtp& StartSync();
    SFtp& StartOpenDir(const String& path);
    SFtp& StartListDir(DirList& list, Gate<DirEntry&> progress = Null);
    SFtp& StartListDir(const String& path, DirList& list, Gate<DirEntry&> progress = Null);
    SFtp& StartMakeDir(const String& path, long mode);
    SFtp& StartRemoveDir(const String& path);
    SFtp& StartMakeLink(const String& orig, const String& link)                      { StartSymLink(orig, const_cast<String*>(&link), LIBSSH2_SFTP_SYMLINK); }
    SFtp& StartReadLink(const String& path, String& target)                          { StartSymLink(path, &target, LIBSSH2_SFTP_READLINK); }
    SFtp& StartRealPath(const String& path, String& target)                          { StartSymLink(path, &target, LIBSSH2_SFTP_REALPATH); }

    SFtp& StartPut(const Stream& in, Gate<int64, int64> progress = Null);
    SFtp& StartPut(const Stream& in, const String& path, unsigned long flags,
                                long mode, Gate<int64, int64> progress = Null);
    SFtp& StartGet(Stream& out, Gate<int64, int64> progress = Null);
    SFtp& StartGet(Stream& out, const String& path, unsigned long flags, long mode,
                                Gate<int64, int64> progress = Null);

    bool Open(const String& path, unsigned long flags, long mode)                   { StartOpen(path, flags, mode); return Execute(); }
    bool Close()                                                                    { StartClose(); return Execute(); }
    bool Remove(const String& path)                                                 { StartRemove(path); return Execute(); }
    bool Rename(const String& oldpath, const String& newpath)                       { StartRename(oldpath, newpath);return Execute(); }
    bool SetStat(const SFtpAttrs& attrs)                                            { StartSetStat(attrs); return Execute(); }
    bool GetStat(SFtpAttrs& attrs)                                                  { StartGetStat(attrs); return Execute(); }
    bool Sync()                                                                     { StartSync(); return Execute(); }
    bool OpenDir(const String& path)                                                { StartOpenDir(path); return Execute(); }
    inline String GetDir() const                                                    { return dir; }
    bool ListDir(SFtp::DirList& list, Gate<DirEntry&> progress = Null)              { StartListDir(list, progress); return Execute(); }
    bool ListDir(const String& path, DirList& list, Gate<DirEntry&> progress = Null){ StartListDir(path, list, progress); return Execute(); }
    bool MakeDir(const String& path, long mode)                                     { StartMakeDir(path, mode); return Execute(); }
    bool RemoveDir(const String& path)                                              { StartRemoveDir(path); return Execute(); }
    bool MakeLink(const String& orig, const String& link)                           { StartMakeLink(orig, link); return Execute(); }
    bool ReadLink(const String& path, String& target)                               { StartReadLink(path, target); return Execute(); }
    bool RealPath(const String& path, String& target)                               { StartRealPath(path, target); return Execute(); }

    bool Put(const Stream& in, Gate<int64, int64> progress = Null)                  { StartPut(in, progress); return Execute(); }
    bool Put(const Stream& in, const String& path, unsigned long flags,
                            long mode, Gate<int64, int64> progress = Null)          { StartPut(in, path, flags, mode, progress); return Execute(); }
    bool Get(Stream& out, Gate<int64, int64> progress = Null)                       { StartGet(out, progress); return Execute(); }
    bool Get(Stream& out, const String& path, unsigned long flags,
                            long mode, Gate<int64, int64> progress = Null)          { StartGet(out, path, flags, mode, progress); return Execute(); }

    bool Seek(int64 position);
    int64 Tell();

    SFtp();
    SFtp(Ssh& session) : SFtp() { Session(session); }
    virtual ~SFtp() { CleanUp(); }

protected:
    virtual void StartInit() override;
    virtual void StartStop() override;
    virtual void CleanUp()   override;

private:
    void StartFStat(SFtpAttrs& attrs, bool set);
    void StartSymLink(const String& path, String* target, int type);

    SFtpSession* session;
    SFtpHandle*  handle;
    DirEntry     dir_entry;
    String       dir;
};

class Channel : public SshSubsystem {
public:
    SshChannel* GetChannel() const { return channel; }

    Channel& StartRequest(const String& request, const String& params = Null);
    Channel& StartExec(const String& cmdline);
    Channel& StartShell();
    Channel& StartSubsystem(const String& subsystem);
    Channel& StartTerminal(const String& term, int width, int height);
    Channel& StartSetEnv(const String& variable, const String& value);
    Channel& StartRead(Stream& in, int64 size, Gate<int64, int64> progress = Null);
    Channel& StartWrite(Stream& out, int64 size, Gate<int64, int64> progress = Null);
    Channel& StartClose();
    Channel& StartCloseWait();
    Channel& StartSendEof();
    Channel& StartRecvEof();
    Channel& StartSendRecvEof();
    Channel& StartReadStdErr(Stream& err);
    Channel& StartWriteStdErr(Stream& err);
    Channel& StartGetExitCode();
    Channel& StartGetExitSignal();

    bool Request(const String& request, const String& params = Null)        { StartRequest(request, params); return Execute(); }
    bool Exec(const String& cmdline)                                        { StartExec(cmdline); return Execute(); }
    bool Shell()                                                            { StartShell(); return Execute(); }
    bool Subsystem(const String& subsystem)                                 { StartSubsystem(subsystem); return Execute(); }
    bool Terminal(const String& term, int width, int height)                { StartTerminal(term, width, height); return Execute(); }
    bool SetEnv(const String& variable, const String& value)                { StartSetEnv(variable, value); return Execute(); }
    bool Read(Stream& in, int64 size, Gate<int64, int64> progress = Null)   { StartRead(in, size, progress); return Execute(); }
    bool Write(Stream& out, int64 size, Gate<int64, int64> progress = Null) { StartWrite(out, size, progress); return Execute(); }
    bool Close()                                                            { StartClose(); return Execute(); }
    bool CloseWait()                                                        { StartCloseWait(); return Execute(); }
    bool SendEof()                                                          { StartSendEof(); return Execute(); }
    bool RecvEof()                                                          { StartRecvEof(); return Execute(); }
    bool SendRecvEof()                                                      { StartSendRecvEof(); return Execute(); }
    bool ReadStdErr(Stream& err)                                            { StartReadStdErr(err); return Execute(); }
    bool WriteStdErr(Stream& err)                                           { StartWriteStdErr(err); return Execute(); }
    int  GetExitCode();
    String GetExitSignal();

    Channel();
    Channel(Ssh& session) : Channel() { Session(session); }
    virtual ~Channel() { CleanUp(); }

protected:
    virtual void StartInit() override;
    virtual void StartStop() override;
    virtual void CleanUp()   override;

    bool DataRead(int id, Stream& out, int64 size, Gate<int64, int64> progress = Null);
    bool DataWrite(int id, Stream& in, int64 size, Gate<int64, int64> progress = Null);

    SshChannel* channel;
    int code;
    StringBuffer signal;
};

class Scp : public Channel {
public:
    Scp& StartGet(Stream& out, const String& path, Gate<int64, int64> progress = Null);
    Scp& StartPut(Stream& in, const String& path, long mode, Gate<int64, int64> progress = Null);

    bool Get(Stream& out, const String& path, Gate<int64, int64> progress = Null)                        { StartGet(out, path, progress); return Execute(); }
    bool Put(Stream& in, const String& path, long mode, Gate<int64, int64> progress = Null)              { StartPut(in, path, mode, progress); return Execute(); }
    inline bool operator()(Stream& out, const String& path, Gate<int64, int64> progress = Null)          { return Get(out, path, progress); }
    inline bool operator()(Stream& in, const String& path, long mode, Gate<int64, int64> progress = Null){ return Put(in, path, mode, progress); }

    Scp()                     { ClearStat(); type = Type::SCP; }
    Scp(Ssh& session) : Scp() { Session(session); }
    virtual ~Scp()            { ClearStat(); }

private:
    inline void ClearStat()   { Zero(file_stat); }
    libssh2_struct_stat file_stat;
};

class Exec : public Channel {
public:
    Exec& StartExecute(const String& cmdline, Stream& out, Stream& err);

    int Execute(const String& cmdline, Stream& out, Stream& err);
    int operator()(const String& cmdline, Stream& out, Stream& err)    { return Execute(cmdline, out, err); }

    Exec()                      { type = Type::EXEC; }
    Exec(Ssh& session) : Exec() { Session(session); }
    virtual ~Exec() {}

private:
    void GetResult();
};

Tuple<int, String> Ssh2LibError(SshSession* session, int code, const char* msg);
}
#endif
